;; Setup breeds and variables

breed [mounds mound]
breed [caves cave]

caves-own [nCrickets] ;; we count crickets per cave as a single variable versus a breed to avoid overwhelming
                      ;; the simulation, population based on energy intake and standard ecological growth models
                      
mounds-own [
  nForagers       ;; similar to our use of crickets as a variable to caves above
  maxdist-mound   ;; maximum foraging distance of the ants from the center of the mound
  aged            ;; this is the age, in weeks, of a mound - we did not use it in our simulations though
  ]

patches-own [
  coverType               ;; covertype variable is based on our coding from ERDAS, one of four cover types
                          ;; based on foliage and energy available to the two species
  available-energy-cave   ;; amount of patch-energy that is available to cricket foragers
  available-energy-mound  ;; amount of patch-energy that is available to fire ant foragers
  maximum-energy          ;; maximum amount of patch-energy a patch can hold, set by coverType
  patch-energy            ;; amount of energy held in a patch that is useful to our two species
                          ;; not all energy is available to the species every tick, only a fraction
  dist-mound              ;; variable holds distance of patch to mound to be used in energy gathering equation
  dist-cave               ;; variable holds distance of patch to cave to be used in energy gathering equation
  probestablish           ;; coefficient of likelihood a new mound will take root within a patch
  occupied                ;; keeps tabs of how many mounds occupy a patch
  ]

globals [
  mapname                 ;; variable to hold name of file to import into View (likely to be called explicitly in future)
  disturbedColor          ;; color of patches with coverType = disturbed
  grassColor              ;; color of patches with coverType = grass
  lowCoverColor           ;; color of patches with coverType = lowCover
  denseCoverColor         ;; color of patches with coverType = denseCover
  weekNumber              ;; holds current week number - one tick = one week
  yearNumber              ;; holds current year number - one year = 52 weeks
  season                  ;; holds current season based on average daily temperature
  Resources-in-Disturbed-Area   ;; initial patch-energy of patches with coverType = disturbed
  Resources-in-Grass-Areas      ;; initial patch-energy of patches with coverType = grass
  Resources-in-Low-Cover        ;; initial patch-energy of patches with coverType = lowCover
  Resources-in-Dense-Cover      ;; initial patch-energy of patches with coverType = denseCover
  nmounds                 ;; count of total mounds
  bscrick0                ;; variable to hold count of crickets for cave 0, used in BehaviorSpace runs
  bscrick1                ;; variable to hold count of crickets for cave 1, used in BehaviorSpace runs
  bscrick2                ;; variable to hold count of crickets for cave 2, used in BehaviorSpace runs
  bscrick3                ;; variable to hold count of crickets for cave 3, used in BehaviorSpace runs
  ]

;; 
;; ALSO SETS UP CAVES AND MOUNDS
;; probestablish - desirability of mounds to propagate to this area during flight



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: Setup
; Purpose: To build the View using an imported image that has already been color coded by ERDAS Imagine and
; then assign variables to the patches, create the initial caves in fixed locations and mound along the road 
; in the SW corner of the View
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to Setup
  
  clear-all
  set mapname "landscape.png" ;we used a 300*300 image and imported into a 100*100 world to show more of the landscape
                               ;and to illustrate the clustering of mounds over time in disturbed patches   
  
  ;the next eight statements assign values to the global variables
  ;we used variables so that we can make changes easier if a color or energy variant led to unexpected results
  ;and it makes the code easier to understand
  
  set disturbedColor red               
  set Resources-in-Disturbed-Area 15
  
  set grassColor yellow 
  set Resources-in-Grass-Areas 65
  
  set lowCoverColor brown
  set Resources-in-Low-Cover 45
  
  set denseCoverColor lime
  set Resources-in-Dense-Cover 25
  
  ;we use a week and year counter, along with a season monitor, to track time in human concepts
  ;one tick = one week and the seasons are based upon week number, decided by average weekly temps for the area
  set weekNumber 1
  set yearNumber 0
  set season "Winter"
  
  ;the actual import of the image file and here we assign shapes to our breeds
  import-pcolors mapname
  set-default-shape mounds "bug" 
  set-default-shape caves "star"
  
   
  ; the variable definitions above explain the usage of the assignments below
  ; this is the only place that maximum-energy is set for a patch
  ; also, although coverType is assigned by patch-color here, this flips during execution
  ; where the coverType will dictate the patch color, besides viewing options increasing
  ; this allows us to change a patch coverType if a patch gets disturbed 
  ask patches [
    if pcolor = 14.9 [set patch-energy Resources-in-Disturbed-Area
                      set maximum-energy Resources-in-Disturbed-Area
                      set coverType "Disturbed Area"
                      set pcolor disturbedColor
                      set probestablish 0.4
                      set occupied 0]
    if pcolor = 44.9 [set patch-energy Resources-in-Grass-Areas
                      set maximum-energy Resources-in-Grass-Areas
                      set coverType "Grass Area"
                      set pcolor grassColor
                      set probestablish 0.02
                      set occupied 0]
    if pcolor = 44.1 [set patch-energy Resources-in-Low-Cover
                     set maximum-energy Resources-in-Low-Cover
                     set coverType "Low Cover"
                     set pcolor lowCoverColor
                     set probestablish 0.005
                     set occupied 0]
    if pcolor = 62.1 [set patch-energy Resources-in-Dense-Cover
                      set maximum-energy Resources-in-Dense-Cover
                      set coverType "Dense Cover"
                      set pcolor denseCoverColor
                      set probestablish 0.001
                      set occupied 0]
  ]
    
  ; since our world doesn't wrap and we are going to have to run some sensitivity tests, we wanted
  ; to have consistent locations for caves, not at the edge of the world and not too close to each other
  ; to avoid crickets competing with each other from two independent caves   
    
  create-caves 1 [
    set color blue
    set size 4
    set nCrickets nCrickets-in-cave
    setxy 46 84
  ]
  
  create-caves 1 [
    set color blue
    set size 4
    set nCrickets nCrickets-in-cave
    setxy 20 52
  ]
  
  create-caves 1 [
    set color blue
    set size 4
    set nCrickets nCrickets-in-cave     
    setxy 53 13
  ]
   
  create-caves 1 [
    set color blue
    set size 4
    set nCrickets nCrickets-in-cave
    setxy 86 40
  ]  
  
  ; we have a chooser to allow 3 initial roudside mounds, without it, the run is caves only
  if setup-mounds [ SetupAddMoundsRoad ]    

end ;;; End Setup


  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: SetupAddMoundsRoad
; Purpose: to add 3 fixed location fire ant mounds, with the number of foragers determined by the slider on interface
; to the road (red patches) on the SW corner of the View
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to SetupAddMoundsRoad
 
   create-mounds 1 [
      set color orange
      set size 1
      set nForagers nForagers-in-mounds
      setxy 2 22
    ]
    
   create-mounds 1 [
      set color orange
      set size 1
      set nForagers nForagers-in-mounds
      setxy 6 10
    ]
    
   create-mounds 1 [
      set color orange
      set size 1
      set nForagers nForagers-in-mounds
      setxy 9 1
    ]
end  ;;; end SetupAddMoundsRoad

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END SETUP PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: Go
; Purpose: Main running procedure, order of business is for each breed to forage, 
; propagate and/or die, raid and keep the energy cycling through the patches and 
; plot data - mostly commented in procedure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
 
  ask mounds [
    moundForage    ; procedure for bringing in energy from patches to mound and changing forager population
    if weekNumber = 20 [ moundReproduce ]  ; procedure to allow mound reproduction seasonally
    if Raiding [ moundRaid ] ; if the user asks for raiding, mounds will raid caves when cave entrances are within mound foraging range
    ;moundAge - not completed
    moundDeath
    moundPlot
   ]

   
ask caves [
   caveDeath  ; kills caves if they get too low in population
   caveForage            ; the main cave procedure, calculating growth rates etc.
   cavePlotCrickets
   if auto-kill [
     if weekNumber = 42 [moundBeKilled] 
   ]             
]
   
ask patches [ 
  patchAddEnergy  ; patches gain energy over time in order to grow
  patchChangeColor    ; changes patch color in View, depending on chooser choice
  patchResetOccupied  ; reset Occupied variable between ticks to avoid summing inadvertantly                        
   ]

set nmounds count mounds ; tracks total mound count for BS runs


tick
ObserveMoveTime

if BehaviorSpaceRun [       ; needed global variables to track crickets per cave at end of BS run, probably replaceable with to-report
 set bscrick0 round( [ncrickets] of cave 0)
 set bscrick1 round( [ncrickets] of cave 1)
 set bscrick2 round( [ncrickets] of cave 2)
 set bscrick3 round( [ncrickets] of cave 3)
]

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END GO PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; BEGIN MOUND PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundForage
; Purpose: Using a population growth model, we build mound activity based on their intake of energy from
; patches within a foraging radius between 20-30m (depending upon mound population)
; the surrounding available energy impacts the mound carrying capacity based on K-sensivity and maximum
; carrying capacity. The energy taken in is translated into increased number of forager ants and
; the ability to propagate every spring
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundForage

  let r (.02423)   ; Tschinkel  1988 - intrinsic rate of growth of RIFA
  let kunlim (250000) ; Tschinkel 1988 - maximum population size of a single mound
  let SensMoundK (0.1) ; sensitivy of mound to local energy and environment
  let energy-gained 0
  set maxdist-mound 3 ; regardless of size, mounds typically send foragers out to 30m, this is more dependent upon variables
                      ; outside the scope of this model
  
  ; we ask all the patches in a 30m radius of the mound to report their distance from the mound
  ; then track how many different mounds can have foraging ants on that patch (occupied)
  ; and set the amount of energy available to that mound given the distance above and the energy in the patch 
  ; the next statement sets the energy gained for the mound, which will be used to impact carrying capacity below
  ; next set the new patch energy after ants forage, leaving less for crickets
  ; and finish up with a call to set the color of patches based on patch-energy if Consumed-Energy is selected                   
  ask patches in-radius maxdist-mound [ 
    set dist-mound distance myself      
    set occupied (occupied + 1)         
    set available-energy-mound ( 0.1 * patch-energy / (dist-mound + 1)) 
    set energy-gained ( energy-gained + available-energy-mound )
    set patch-energy ( patch-energy - available-energy-mound )
     if View = "Consumed-Energy" [
       ifelse coverType = "Disturbed Area" [set pcolor scale-color disturbedColor patch-energy 0 100 ] [
       ifelse coverType = "Grass Area" [set pcolor scale-color grassColor patch-energy 0 100 ] [
       ifelse coverType = "Low Cover" [set pcolor scale-color lowCoverColor patch-energy 0 100 ] [
       if coverType = "Dense Cover" [set pcolor scale-color denseCoverColor patch-energy 0 100 ] ]]]
      ] 
   ]  
  
  ; the first equation relates the local energy taken in to the carrying capacity of this particular mound
  ; the second sets the number of foragers based on the Lotva-Voltaire model and newly set k 
  let k (kunlim * (1 - 2.71828183 ^ (- SensMoundK * energy-gained ))) 
  set nForagers (nForagers + (r * nForagers - r * nForagers * nForagers / k))
  
end  ;; end moundForage procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundReproduce
; Purpose: Allows mound of 5000 foragers or more to pit our a new young mound that may or may not survive
; dependent upon finding a decent landing spot
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundReproduce  ;; MOUND PROCEDURE
   
   if nForagers < 200 [die] ;small mounds die due to lack of ability to bring in enough energy to queen
      
   ; this procedure creates a new "youngmound" and places it on the map based on nearby mounds and patch-type
   ; only big mounds reproduce, it takes about 2.5 years for a mound to get to 5000 foragers
   ; large mounds then send out 1 new mound, which will try and find a desirable location - typically disturbed areas
   ; if it cannot find one, there is a slight chance it will take to another area (based on the probestablsh variable of the patch)
   ; or it will die. The loop in this section allows the recreation of clustering, meaning that a new queen will try an area close
   ; first and then go further and further - until they succeed of die
   ; this makes our maximum jump from a mound up to 25 * 5 or 1250 m, rare but possible with wind - Drees
   if nForagers > 5000 [      
       hatch-mounds 1 [                            
         rt random-float 360
         let count-loop 0 
         loop [
             fd random-float 5
             set count-loop (count-loop + 1)  ; counter needed so mounds can't go forever
             if count-loop > 25 [die]
             if  occupied = 0 [if random-float 1 < probestablish [set nForagers 200 stop]]
          ]
       ]  
    ]

end ;; end moundReproduce procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundRaid
; Purpose: The idea is that when surface temperatures are above 85F, fire ants start putting the mound deeper
; and relying more heavily on foraging tunnels, increasing the likelihood of entering cooler caves and killing
; young crickets or taking eggs. We have them snagging 100 crickets per raid, the number of times a cave is "raided" 
; will depend upon the number of mounds in raiding distance (which is the foraging distance of the mound).
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundRaid
   if ( Season = "Summer" and  (random 5) = 2 ) [
    ask caves in-radius maxdist-mound [ 
      ifelse nCrickets >= 101 [set nCrickets (nCrickets - 100)][set nCrickets 0]
    ]]     
end  ;;; End moundRaid Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundBeKilled
; Purpose: This is a management strategy of manually killing mounds by spotting them and killing them
; with hot water treatment - it has a 60% success rate within a 50m radius of the cave
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to moundBeKilled

    let killnumber round((count mounds in-radius 5) * .6)
    ask n-of killnumber mounds in-radius (5) [die]
  
end ;; End moundBeKilled Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundDeath
; Purpose: If a mound cannot gather enough energy, it will die. This is set arbitrarily at his point
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundDeath  

   if nForagers < 200 [ 
     die 
     ]  

end  ;;;end moundDeath Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundAge
; Purpose: Since some functions are dependent upon aging...
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundAge
 
   set aged (aged + 1)

end  ;;; end moundAge procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: moundPlot
; Purpose:  Plots number of foragers per mound, will have all mounds on plot
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to moundPlot
 
  set-current-plot "Foragers"
  set-current-plot-pen "Foragers"
  plot (nForagers)

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END MOUND PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; BEGIN CAVE PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: caveForage
; Purpose: Using a population growth model, we build mound activity based on their intake of energy from
; patches within a foraging radius between 50-100m (depending upon cricket population)
; the surrounding available energy impacts the cave carrying capacity based on K-sensivity and maximum
; carrying capacity. The energy taken in is translated into variable number of crickets
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to caveForage 

  let r (r-cave)  ; we have a rough idea of the intrinsic growth rate from other species of crickets, though this
                  ; should be better defined through external lab work
  let kunlim (kunlim-cave) ; the average is 5000 from research on the base, but it varies a lot from cave to cave
                           ; we believe this is an important area to pursue through sensitivity analysis and field work
  let maxdist 10; 
    ;;sets a maximum foraging distance for caves, recorded as 100m : Taylor 2002
  let energy-gained-cave 0  
  
  ask patches in-radius maxdist [ 
    set dist-cave distance myself
    set available-energy-cave (0.1 * patch-energy / (dist-cave + 1)) 
    set energy-gained-cave ( energy-gained-cave + available-energy-cave )
    set patch-energy ( patch-energy - available-energy-cave)
    if View = "Consumed-Energy" [
       ifelse coverType = "Disturbed Area" [set pcolor scale-color disturbedColor patch-energy 0 100 ] [
       ifelse coverType = "Grass Area" [set pcolor scale-color grassColor patch-energy 0 100 ] [
       ifelse coverType = "Low Cover" [set pcolor scale-color lowCoverColor patch-energy 0 100 ] [
       if coverType = "Dense Cover" [set pcolor scale-color denseCoverColor patch-energy 0 100 ] ]]]
     ]
   ]

  let k (kunlim * (1 - 2.71828183 ^ (- SensCaveK * energy-gained-cave )))  
  ; adjusts the carrying capacity of a cave (k) to be an exponential function
  ; of the energy gained from the environment
                                       
  set nCrickets (nCrickets + (r * nCrickets - r * nCrickets * nCrickets / k))
 
end ;;; End caveForage Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: caveDeath
; Purpose: If a cave cannot gather enough energy, it will die. This is set arbitrarily at this point
; We also turn this off for the Behavior Space runs to avoid errors
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to caveDeath

  if nCrickets <= 300  [die]
  
end ;;; End caveDeath Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedure: cavePlotCrickets
; Purpose:  Plots number of crickets per cave, will have all caves on plot
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to cavePlotCrickets

  set-current-plot "Crickets"
  set-current-plot-pen "Crickets"
  plot (nCrickets)

end  ;;;End cavePlotCrickets Procedure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END CAVE PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; BEGIN PATCH PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to patchAddEnergy  
  ;; every tick, 20% of the patches will gain one unit of energy 
 
   if ( patch-energy < maximum-energy AND (random 5 ) = 2 )
     [  set patch-energy (patch-energy + 1) ]
   
end

to patchChangeColor
  ;; this allows the view to change, each patch will show it's energy by color
  ;; useful to see the variance of available energy in foraging ranges
 
 if View = "Patch-Energy" [
    set pcolor patch-energy
  ]
    
end

to patchResetOccupied
  ;; we only allow new mounds to start in mounds that are "unoccupied"
  
  set occupied 0

end

to patchResetColor 
;; procedure to reset original map image behind caves and mounds when View changes distort map

ifelse coverType = "Disturbed Area" [set pcolor disturbedColor] [
  ifelse coverType = "Grass Area" [set pcolor grassColor] [
    ifelse coverType = "Low Cover" [set pcolor lowCoverColor] [
      if coverType = "Dense Cover" [set pcolor denseCoverColor] 
]]]
 
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END PATCH PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; BEGIN MISC PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to observeMoveTime ;;changes seasons based on week number, counts 52 weeks in a year, increments every tick

  if weekNumber >= 1 and weekNumber <= 8 [ set season "Winter" ]
  if weekNumber >= 9 and weekNumber <= 23 [ set season "Spring" ]
  if weekNumber >= 24 and weekNumber <= 34 [ set season "Summer" ]
  if weekNumber >= 35 and weekNumber <= 48 [ set season "Fall" ] 
  if weekNumber >= 49 and weekNumber <= 52 [ set season "Winter" ]

  ifelse weeknumber >= 52 [set yearNumber yearNumber + 1
                         set weekNumber 1]
        [set weekNumber weekNumber + 1]

end ;;; END observeMoveTime

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; END MISC PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;










@#$#@#$#@
GRAPHICS-WINDOW
210
15
620
446
-1
-1
4.0
1
10
1
1
1
0
0
0
1
0
99
0
99
1
1
1
ticks

INPUTBOX
16
58
180
118
nCrickets-in-cave
2000
1
0
Number

PLOT
645
16
900
176
Crickets
Time
Number
0.0
10.0
0.0
500.0
true
true
PENS
"Crickets" 1.0 0 -7500403 false

INPUTBOX
16
263
180
323
nForagers-in-mounds
2000
1
0
Number

BUTTON
212
457
311
517
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
17
170
179
203
Raiding
Raiding
1
1
-1000

CHOOSER
330
456
495
501
View
View
"Consumed-Energy" "Patch-Energy"
0

MONITOR
742
413
813
458
Week
weekNumber
0
1
11

MONITOR
647
413
716
458
Year
yearNumber
0
1
11

MONITOR
832
413
906
458
Season
season
0
1
11

BUTTON
508
456
616
489
Restore Map
patchResetColor
NIL
1
T
PATCH
NIL
NIL
NIL
NIL

BUTTON
18
14
183
47
Setup Caves
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

PLOT
644
183
903
343
Foragers
Time
Number
0.0
10.0
0.0
10.0
true
false
PENS
"Foragers" 1.0 0 -16777216 false

SLIDER
15
332
182
365
r-cave
r-cave
0.001
.1
0.01
.001
1
NIL
HORIZONTAL

SLIDER
14
375
183
408
kunlim-cave
kunlim-cave
1000
10000
5000
100
1
NIL
HORIZONTAL

SLIDER
14
418
184
451
SensCaveK
SensCaveK
.01
.2
0.05
.01
1
NIL
HORIZONTAL

TEXTBOX
18
462
180
537
Standard Defaults\nr-cave = 0.01\nkunlim-cave = 5000\nSensCaveK = 0.05
11
0.0
1

SWITCH
17
215
180
248
Auto-Kill
Auto-Kill
1
1
-1000

SWITCH
17
127
180
160
setup-mounds
setup-mounds
0
1
-1000

MONITOR
646
355
766
400
Count of Mounds
count mounds
0
1
11

MONITOR
793
355
901
400
Count of Caves
count caves
0
1
11

SWITCH
738
476
908
509
BehaviorSpaceRun
BehaviorSpaceRun
1
1
-1000

@#$#@#$#@
WHAT IS IT?
-----------
This is a model demonstrating impacts of an invasive species of fire ants (Solenpsis invicta) on cave crickets, their ability to forage and bring back bio-energy to caves in Fort Hood, Texas. The potentially endangered fauna within the cave are dependent upon the cave crickets for energy, so reducing the crickets that bring energy into the cave should also impact the health of the fauna within the cave.




HOW IT WORKS
------------
We use a standard Lotka-Volterra logistic equation as our base for the two species in reaching an equilibrium with the environment. Since the initial species are not in competition with each other due to distance, their populations are either capped by the available energy impacting their carrying capacity (mounds) or the carrying capacity variable represents the maximum size of the population regardless of available energy available (caves).

As the mounds enter the foraging area of the crickets, the amount of available energy available to the crickets will decrease and the carrying capacity will the be impacted by the lower amounts of energy available to foraging crickets.



HOW TO USE IT
-------------
Setup Caves will import a color coded map of Fort Hood, each color represents a cover typ

We have also the following selections from the interface:

1. [ setup-mounds  switch ] Ability to run the model without ants present to note the equilibrium times of the cricket populations.

2. [ raiding switch ] Ability to turn fire ant raiding of crickets within the cave on/off. Raiding occurs 20% of the time during the summers if the foraging range of a mound touches a cave entrance.

3. [ auto-kill switch ] Ability to set up a fire ant mound management technique that will annually kill 60% of the mounds within a 50m radius of the cave entrance.

4. [ nCrickets-in-Caves ] [ nForagers-in-Mounds ] Input boxes for initial levels of species in held by their breeds.

5. [ r-cave slider ] Allows variations in the intrinsic rate of growth of crickets in the cave. This value is not certain from literature.

6. [ kunlim-cave slider ] Allows variations in the carrying capacity of crickets in the cave. This value is not certain from literature and varies on a cave by cave basis (Taylor)

7. [ SensCaveK slider ] Allows variations in the sensitivity of crickets in the cave to changes in the environmental energy. This value is not certain from literature and may vary from cave to cave.

8. Chooser to change view from Consumed Energy in patches surrounding the breeds within the foraging range of each breed or Patch Energy, which shows the change of patch color in relation to the amount of patch-energy. The Restore Map refreshed the map with the original color in case of ghosting from switching views.

THINGS TO NOTICE
----------------
After running the simulation a few times, yu will notice that "big" caves, with a large carrying capacity, are impacted marginally - especially if the k-sensitivity is low. Raiding does not seem to have a long term impact for the most part. But over longer periods of time, the reduced carrying capacities seem to have a greater impact, especially without any type of control over the fire ants mounds.


THINGS TO TRY
-------------
Since Fort Hood has a lot of "different" caves in terms of size, depth, number of entrances, internal humidity, etc. The three sliders impacting r, k and k-sensitivity are crucial for determining impacts on different types of caves. A small, shallow cave (low k, high k-sens) may show different responses to a large, deep cave that can hold lots of crickets (high k, low k-sens).


EXTENDING THE MODEL
-------------------
We have a few ideas for extending the model, but we would first want to gain more precise information regarding the r, k and k-sens values for caves/crickets in this area. With that information, we can extend the model to include:

0. Completing the Lotka-Volterra for species in competition for resources

1. The differences is various fire ant management techniques.
2. The impact of these techniques on the re-infestation of fire ants or carrying capacity of the crickets.
3. Creating differing cave environments across the four caves instead of having identical caves.
4. Adding patch-energy fluctuations to see impacts of energy constraints on survivability of species in competition.



CREDITS AND REFERENCES
----------------------
Steven J Taylor, from the Illinois Natural History Survey, was instrumental in developing some of these sub models based upon his collected data and personal observations.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Caves" repetitions="1" runMetricsEveryStep="false">
    <setup>setup-fixed
</setup>
    <go>go</go>
    <timeLimit steps="1560"/>
    <metric>bscrick1</metric>
    <metric>bscrick2</metric>
    <metric>bscrick3</metric>
    <metric>bscrick4</metric>
    <metric>nmounds</metric>
    <enumeratedValueSet variable="SensCaveK">
      <value value="0.02"/>
      <value value="0.05"/>
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="kunlim-cave">
      <value value="1000"/>
      <value value="5000"/>
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Raiding">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="auto-kill">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="setup-mounds">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
